/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/
package builder

import (
	"fmt"
	"time"

	"jobarranger2/src/libs/golibs/common"
	"jobarranger2/src/libs/golibs/config_reader/server"
	"jobarranger2/src/libs/golibs/database"
)

// Jobnet related query builders starts here
func GetJobnetStatusChangeQuery(innerJobnetId uint64) {
}

// Jobnet related query builders ends here
// ************************************************************************************************
// ************************************************************************************************

// Job related query builders starts here
func GetJobStatusChangeQuery(
	innerJobId uint64,
	status common.StatusType,
	methodFlag common.JobRunMethod,
	runCnt int,
	start int, end int,
) (string, string) {
	// functionName := "getJobStatusChangeQuery"
	// log.Printf("[DEBUG] In %s() innerJobID: %d, status: %d\n", functionName, innerJobId, status)

	ts := time.Now().Unix()
	timestampStr := fmt.Sprintf("%d", ts)

	var strMethodFlag, strRunCnt, strStart, strEnd string

	switch methodFlag {
	case -1:
		strMethodFlag = ""
	default:
		strMethodFlag = fmt.Sprintf(", method_flag=%d", methodFlag)
	}

	switch runCnt {
	case 0:
		strRunCnt = ""
	default:
		strRunCnt = fmt.Sprintf(", run_count = %d", runCnt)
	}

	switch start {
	case 0:
		strStart = ", start_time = 0"
	case 1:
		strStart = fmt.Sprintf(", start_time = %s", timestampStr)
	default:
		strStart = ""
	}

	switch end {
	case 0:
		strEnd = ", end_time = 0"
	case 1:
		strEnd = fmt.Sprintf(", end_time = %s", timestampStr)
	default:
		strEnd = ""
	}

	// Simulated SQL (skip DBselect lock and actual DBexecute)
	lockSQL := fmt.Sprintf("select status from ja_2_run_job_table where inner_job_id = %d for update", innerJobId)

	updateSQL := fmt.Sprintf(
		"update ja_2_run_job_table set status = %d%s%s%s%s where inner_job_id = %d",
		status, strMethodFlag, strRunCnt, strStart, strEnd, innerJobId,
	)

	// Simulate success
	return lockSQL, updateSQL
}

func GetRunJobMethodFlagChangeQuery(innerJobId uint64, methodFlag common.JobRunMethod) (string, string) {
	lockSQL := fmt.Sprintf("select method_flag from ja_2_run_job_table "+
		"where inner_job_id = %d for update", innerJobId)

	updateSQL := fmt.Sprintf("UPDATE ja_2_run_job_table "+
		"SET method_flag = %d "+
		"where inner_job_id = %d",
		methodFlag, innerJobId,
	)
	return lockSQL, updateSQL
}

func GetRunJobRetryCountIncrementQuery(innerJobId uint64, retryCount int) string {
	return fmt.Sprintf("UPDATE ja_2_run_job_table "+
		"SET retry_count = %d "+
		"where inner_job_id = %d",
		retryCount+1, innerJobId,
	)
}

func GetRunJobVariableInsertQuery(runJobVaribaleData common.RunJobVariableTable) string {
	var query string

	switch server.Options.DBType {
	case database.MysqlDBType, database.MariaDBType:
		query = fmt.Sprintf("INSERT INTO %s "+
			"(seq_no, inner_job_id, inner_jobnet_id, before_variable) "+
			"VALUES (%d, %d, %d, '%s') "+
			"ON DUPLICATE KEY UPDATE before_variable = VALUES(before_variable);",
			common.Ja2RunJobVariableTable,
			runJobVaribaleData.SeqNo,
			runJobVaribaleData.InnerJobID,
			runJobVaribaleData.InnerJobnetID,
			runJobVaribaleData.BeforeVariable,
		)

	case database.PostgresDBType:
		query = fmt.Sprintf("INSERT INTO %s "+
			"(seq_no, inner_job_id, inner_jobnet_id, before_variable) "+
			"VALUES (%d, %d, %d, '%s') "+
			"ON CONFLICT (seq_no, inner_job_id, inner_jobnet_id) "+
			"DO UPDATE SET before_variable = EXCLUDED.before_variable;",
			common.Ja2RunJobVariableTable,
			runJobVaribaleData.SeqNo,
			runJobVaribaleData.InnerJobID,
			runJobVaribaleData.InnerJobnetID,
			runJobVaribaleData.BeforeVariable,
		)
	}
	return query
}

func GetRunJobVariableDeleteQuery(innerJobId uint64) string {
	return fmt.Sprintf("DELETE FROM %s WHERE inner_job_id = %d",
		common.Ja2RunJobVariableTable, innerJobId)
}

func GetHostLockQuery(host string) (string, string) {

	lockSQL := "select lock_host_name from ja_2_host_lock_table " +
		"where lock_host_name = 'HOST_LOCK_RECORD' for update"

	insertSQL := fmt.Sprintf(
		"insert into ja_2_host_lock_table (lock_host_name) values ('%s')",
		host,
	)

	return lockSQL, insertSQL
}

func GetRunTimeoutTableUpdateAbortFlagQuery(innerJobId uint64, flag bool) (string, string) {
	flagStr := "FALSE"
	if flag {
		flagStr = "TRUE"
	}

	lockSQL := fmt.Sprintf("select abort_flag from ja_2_run_timeout_table where inner_job_id = %d for update", innerJobId)
	updateSQL := fmt.Sprintf("UPDATE ja_2_run_timeout_table SET abort_flag = %s WHERE inner_job_id = %d", flagStr, innerJobId)

	return lockSQL, updateSQL
}
