/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

package main

import (
	"fmt"

	"jobarranger2/src/libs/golibs/builder"
	"jobarranger2/src/libs/golibs/common"
	"jobarranger2/src/libs/golibs/database"
	"jobarranger2/src/libs/golibs/event"
	"jobarranger2/src/libs/golibs/logger/logger"
	"jobarranger2/src/libs/golibs/utils"
)

func ProcessJobnetStatusChange(eventName common.EventName, jobnetStatusChange common.JobnetRunData) error {
	var nextEventData common.EventData
	funcName := "ProcessJobnetStatusChange"

	logger.JaLog("JAFLOW400001", logger.Logging{}, funcName,
		fmt.Sprintf("inner_jobnet_id: %d", jobnetStatusChange.InnerJobnetId))

	nextEventData.Event.Name = eventName
	nextEventData.Event.UniqueKey = common.GetUniqueKey(processName)
	nextEventData.NextProcess.Name = common.JobnetManagerProcess
	nextEventData.NextProcess.Data = jobnetStatusChange

	err := event.CreateNextEvent(nextEventData, jobnetStatusChange.InnerJobnetId, jobnetStatusChange.JobnetID, 0)
	if err != nil {
		logger.JaLog("JAFLOW200000", logger.Logging{}, funcName, "CreateNextEvent", err)
		return fmt.Errorf("CreateNextEvent() failed")
	}
	logger.JaLog("JAFLOW400002", logger.Logging{}, funcName,
		fmt.Sprintf("inner_jobnet_id: %d",
			jobnetStatusChange.InnerJobnetId))
	return nil
}

func ProcessJobnetIconStatusChange(
	conn database.DBConnection,
	processData common.FlowProcessData,
) error {
	var nextEventData common.EventData
	var jobnetRunData common.JobnetRunData
	var runJobData common.RunJobTable
	var runJobnetData common.RunJobnetTable
	var err error
	funcName := "ProcessJobnetIconStatusChange"

	nextEventData.Event.Name = common.EventFlowJobnetIconStatusChange
	nextEventData.Event.UniqueKey = common.GetUniqueKey(processName)
	nextEventData.NextProcess.Name = common.JobnetManagerProcess

	runJobData, err = utils.GetRunJobData(conn, processData.InnerJobId)
	if err != nil {
		return fmt.Errorf("GetRunJobData() failed, err: [%w]", err)
	}
	runJobnetData, err = utils.GetRunJobnetTableDataByInnerJobnetId(conn, runJobData.InnerJobnetID)
	if err != nil {
		return fmt.Errorf("GetRunJobnetTableDataByInnerJobnetId() failed, err: [%w]", err)
	}

	jobnetRunData.JobnetID = runJobnetData.JobnetID
	jobnetRunData.InnerJobnetId = runJobData.InnerJobnetID
	jobnetRunData.InnerJobId = runJobData.InnerJobID
	err = utils.Convert(processData.Data, &jobnetRunData.ActionIconData)
	if err != nil {
		return fmt.Errorf("data is not for jobnet icon status change process, data: %v, error [%w]",
			processData.Data, err)
	}

	nextEventData.NextProcess.Data = jobnetRunData

	lockQuery, updateQuery := builder.GetJobStatusChangeQuery(
		processData.InnerJobId, processData.Status, -1,
		0, 0, 0)
	nextEventData.Queries = append(nextEventData.Queries, lockQuery, updateQuery)

	err = event.CreateNextEvent(nextEventData, processData.InnerJobnetId, processData.JobnetId, 0)
	if err != nil {
		logger.JaLog("JAFLOW200000", logger.Logging{}, funcName, "CreateNextEvent", err)
		return fmt.Errorf("CreateNextEvent() failed")
	}
	logger.JaLog("JAFLOW400008", logger.Logging{}, funcName,
		processData.Status, processData.InnerJobId, processData.InnerJobnetId)

	return nil
}
