<?php
/*
** Job Arranger Manager
** Copyright (C) 2023 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** Licensed to the Apache Software Foundation (ASF) under one or more
** contributor license agreements. See the NOTICE file distributed with
** this work for additional information regarding copyright ownership.
** The ASF licenses this file to you under the Apache License, Version 2.0
** (the "License"); you may not use this file except in compliance with
** the License. You may obtain a copy of the License at
**
** http://www.apache.org/licenses/LICENSE-2.0
**
** Unless required by applicable law or agreed to in writing, software
** distributed under the License is distributed on an "AS IS" BASIS,
** WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
** See the License for the specific language governing permissions and
** limitations under the License.
**
**/

namespace App\Controllers\Api;

use App\Utils\Constants;
use App\Utils\Util;
use Rakit\Validation\Validator;

/**
 * This controller is used to manage user login.
 *
 * @version    6.1.0
 * @since      Class available since version 6.1.0
 */
class LoginApi
{
    const JSON_FAILED_FLAG = "json_failed";
    /**
     * It login user to zabbix API.
     *
     * @return  string could be logged in user info
     * @since      Class available since version 6.1.0
     */
    public static function AuthPathAPI($data)
    {
        
        $url = ZBX_API_ROOT . Constants::ZBX_MAIN_END_POINT;
        $sslVerificationFlag = SSL_VERIFICATION == 1;
        $version = self::checkZbxApiInfo();
        if(!$version){
            return false;
        }
        $check = version_compare($version, "6.0.0",null);
        if($check>=0){
            $arr = array('username' => $data['username'], 'password' => $data['password'], 'userData' => 'true');
        }else{
            return false;
        }
        

        $payload = json_encode(array(
            'jsonrpc' => '2.0',
            'method' => 'user.login',
            'params' => $arr,
            'id' => 1
        ));

        $options = [
            'http' => [
                'header' => "Content-Type: application/json\r\n",
                'method' => 'POST',
                'content' => $payload,
            ],
            'ssl' => [
                'verify_peer' => $sslVerificationFlag,
                'verify_peer_name' => $sslVerificationFlag,
            ],
        ];

        $context = stream_context_create($options);

        // Perform the HTTP request and get the response
        $result = file_get_contents($url, false, $context);
        if ($result === false) {
            return false;
        }
        $json_test = self::isJSON($result);
        if($json_test === false){
            return self::JSON_FAILED_FLAG;
        }
        return $result;
    }

    public static function checkAuthentication($sessionId)
    {
        $url = ZBX_API_ROOT . Constants::ZBX_MAIN_END_POINT;
        $sslVerificationFlag = SSL_VERIFICATION == 1;
        $arr = array('sessionid' => $sessionId);

        $payload = json_encode(array(
            'jsonrpc' => '2.0',
            'method' => 'user.checkAuthentication',
            'params' => $arr,
            'id' => 1,
        ));

        $options = [
            'http' => [
                'header' => "Content-Type: application/json\r\n",
                'method' => 'POST',
                'content' => $payload,
            ],
            'ssl' => [
                'verify_peer' => $sslVerificationFlag,
                'verify_peer_name' => $sslVerificationFlag,
            ],
        ];

        $directoryName = __DIR__ . "\\jamlocal.crt";
        if (file_exists($directoryName)) {
            $options['ssl'] = [
                'cafile' => $directoryName,
            ];
        }

        $context = stream_context_create($options);

        // Perform the HTTP request and get the response
        $result = file_get_contents($url, false, $context);

        return $result;
    }

    public static function isJSON($json){
        $VD = new validator();
        $json_arr = Util::jsonDecode($json);
        if(!$json_arr){
            return false;
        }
        $validation = $VD->validate($json_arr, [
        'jsonrpc' => 'required',
        'id' => 'required',
        ]);

        if ($validation->fails()) {
            return false;
        }
        if(!isset($json_arr['result'])){
            if(!isset($json_arr['error'])){    
                return false;
            }else{
                return true;
            }
        }
        $validation = $VD->validate($json_arr['result'], [
            'userid' => 'required',
            'username' => 'required',
            'lang' => 'required',
            'refresh' => 'required',
            'attempt_clock' => 'required',
            'rows_per_page' => 'required',
            'timezone' => 'required',
            'roleid' => 'required',
            'type' => 'required',
            'userip' => 'required',
            'debug_mode' => 'required',
            'gui_access' => 'required',
            'sessionid' => 'required',
        ]);
        if ($validation->fails()) {
            return false;
        }
        return true;

    }
    public static function logoutAPI($sessionId)
    {
        $url = ZBX_API_ROOT . Constants::ZBX_MAIN_END_POINT;
        $sslVerificationFlag = SSL_VERIFICATION == 1;
        $payload = json_encode(array(
            'jsonrpc' => '2.0',
            'method' => 'user.logout',
            'params' => [],
            'id' => 1,
            'auth' => $sessionId
        ));

        $options = [
            'http' => [
                'header' => "Content-Type: application/json\r\n",
                'method' => 'POST',
                'content' => $payload,
            ],
            'ssl' => [
                'verify_peer' => $sslVerificationFlag,
                'verify_peer_name' => $sslVerificationFlag,
            ],
        ];

        $context = stream_context_create($options);

        // Perform the HTTP request and get the response
        $result = file_get_contents($url, false, $context);

        return $result;
    }

    public static function checkZbxApiInfo(){
        $url = ZBX_API_ROOT . Constants::ZBX_MAIN_END_POINT;
        $sslVerificationFlag = SSL_VERIFICATION == 1;
        $payload = json_encode(array(
            'jsonrpc' => '2.0',
            'method' => 'apiinfo.version',
            'params' => [],
            'id' => 1,
        ));
        $options = [
            'http' => [
                'header' => "Content-Type: application/json\r\n",
                'method' => 'POST',
                'content' => $payload,
            ],
            'ssl' => [
                'verify_peer' => $sslVerificationFlag,
                'verify_peer_name' => $sslVerificationFlag,
            ],
        ];

        $context = stream_context_create($options);

        // Perform the HTTP request and get the response
        $result = file_get_contents($url, false, $context);
        $VD = new validator();
        $json_arr = Util::jsonDecode($result);
        $validation = $VD->validate($json_arr, [
        'jsonrpc' => 'required',
        'id' => 'required',
        'result' => 'required',
        ]);

        if ($validation->fails()) {
            return false;
        }
        return $json_arr['result'];
    }
}
