/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

package main

import (
	"fmt"
	"time"

	"jobarranger2/src/libs/golibs/common"
)

func jaTimestamp2Str() string {
	return fmt.Sprintf("%d", time.Now().Unix())
}
func jaSetStatusJobnetSummary(innerJobnetId uint64, status common.StatusType, start, end int, queries ...string) []string {
	timestamp := jaTimestamp2Str()
	strStart := buildTimeClause("start_time", start, timestamp)
	strEnd := buildTimeClause("end_time", end, timestamp)
	var updateSql string
	if status == common.StatusEnd || status == common.StatusEndErr {
		deleteSql := fmt.Sprintf("DELETE from ja_2_session_table where inner_jobnet_main_id=%d ", innerJobnetId)
		queries = append(queries, deleteSql)

		updateSql = fmt.Sprintf(
			"update ja_2_run_jobnet_summary_table set status = %d%s%s ,running_job_id = '', running_job_name = '' where inner_jobnet_id = %d",
			status, strStart, strEnd, innerJobnetId,
		)
		queries = append(queries, updateSql)
	} else {
		updateSql = fmt.Sprintf(
			"update ja_2_run_jobnet_summary_table set status = %d%s%s where inner_jobnet_id = %d",
			status, strStart, strEnd, innerJobnetId,
		)
		queries = append(queries, updateSql)
	}
	return queries
}

func jaSetStatusJobnet(innerJobnetId uint64, status common.StatusType, start, end int, queries ...string) []string {
	timestamp := jaTimestamp2Str()
	strStart := buildTimeClause("start_time", start, timestamp)
	strEnd := buildTimeClause("end_time", end, timestamp)

	updateSQL := fmt.Sprintf("update ja_2_run_jobnet_table set status = %d%s%s where inner_jobnet_id = %d", status, strStart, strEnd, innerJobnetId)

	return append(queries, updateSQL)
}

func buildTimeClause(field string, flag int, timestamp string) string {
	switch flag {
	case 0:
		return fmt.Sprintf(", %s = 0", field)
	case 1:
		return fmt.Sprintf(", %s = %s", field, timestamp)
	default:
		return ""
	}
}

func jaSetStatusJobnetWait(innerJobnetId uint64, queries ...string) []string {
	query := fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set load_status = %d where inner_jobnet_id = %d",
		common.JA_SUMMARY_LOAD_STATUS_DELAY, innerJobnetId,
	)
	return append(queries, query)
}

func jaSetStatusJobnetSkip(innerJobnetId uint64, queries ...string) []string {
	now := time.Now().Unix()
	updateSQL := []string{}
	summaryQuery := fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set status = %d, load_status = %d, start_time = %d, end_time = %d where inner_jobnet_id = %d",
		JA_JOBNET_STATUS_END, common.JA_SUMMARY_LOAD_STATUS_SKIP, now, now, innerJobnetId,
	)
	updateSQL = append(updateSQL, summaryQuery)

	jobnetQuery := fmt.Sprintf(
		"update ja_2_run_jobnet_table set status = %d, start_time = %d, end_time = %d where inner_jobnet_id = %d",
		JA_JOBNET_STATUS_END, now, now, innerJobnetId,
	)
	updateSQL = append(updateSQL, jobnetQuery)

	return append(queries, updateSQL...)
}

func jaSetStatusJobnetTimeout(innerJobnetId uint64, jobnet_timeout_run_type int, queries ...string) []string {
	switch jobnet_timeout_run_type {
	case 0:
		//Timeout Warning
		summaryQuery := fmt.Sprintf(
			"update ja_2_run_jobnet_summary_table set job_status = %d, jobnet_abort_flag = %d, jobnet_timeout_flag =1 where inner_jobnet_id = %d",
			JA_SUMMARY_JOB_STATUS_TIMEOUT, jobnet_timeout_run_type, innerJobnetId,
		)
		queries = append(queries, summaryQuery)
	case 1:
		summaryQuery := fmt.Sprintf(
			"update ja_2_run_jobnet_summary_table set status = %d, job_status = %d, jobnet_abort_flag = %d, jobnet_timeout_flag =1 where inner_jobnet_id = %d",
			JA_JOBNET_STATUS_RUN, JA_SUMMARY_JOB_STATUS_ERROR, jobnet_timeout_run_type, innerJobnetId,
		)
		queries = append(queries, summaryQuery)

	}
	return queries

}

func jaSetStatusIconTimeoutWarning(innerJobnetId uint64, queries ...string) []string {
	timeoutFlagQuery := fmt.Sprintf(
		"UPDATE ja_2_run_jobnet_table set timeout_flag=1 where inner_jobnet_id = %d",
		innerJobnetId,
	)
	queries = append(queries, timeoutFlagQuery)

	timeoutWarningQuery := fmt.Sprintf(
		"UPDATE ja_2_run_jobnet_summary_table set job_status = %d where inner_jobnet_id = %d",
		JA_SUMMARY_JOB_STATUS_TIMEOUT,
		innerJobnetId,
	)
	queries = append(queries, timeoutWarningQuery)

	return queries
}

func jaStatusJobnetDelayStart(innerJobnetId uint64, queries ...string) []string {
	now := time.Now().Unix()
	queries = append(queries, fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set update_date = %d, status = %d, job_status = %d, load_status = %d where inner_jobnet_id = %d",
		now, JA_JOBNET_STATUS_RUN, JA_SUMMARY_JOB_STATUS_ERROR, common.JA_SUMMARY_LOAD_STATUS_DELAY, innerJobnetId,
	))

	return queries
}

func jaStatusScheduleJobnetDelete(innerJobnetMainId uint64, queries ...string) []string {
	futureTime := time.Now().Add(5 * time.Minute).Unix()

	deleteSql := fmt.Sprintf(
		"delete from ja_2_run_jobnet_table where inner_jobnet_main_id=%d and scheduled_time > %d",
		innerJobnetMainId, futureTime,
	)
	return append(queries, deleteSql)
}

func jaStatusJobnetHoldSummary(innerJobnetId uint64, queries ...string) []string {
	holdSql := fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set start_pending_flag=%d where inner_jobnet_id=%d and status=%d",
		JA_SUMMARY_START_PENDING_ON, innerJobnetId, JA_JOBNET_STATUS_BEGIN,
	)
	return append(queries, holdSql)
}

func jaStatusJobnetReleaseSummary(innerJobnetId uint64, queries ...string) []string {
	releaseSql := fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set start_pending_flag=%d where inner_jobnet_id=%d and start_pending_flag=%d",
		JA_SUMMARY_START_PENDING_OFF, innerJobnetId, JA_SUMMARY_START_PENDING_ON,
	)
	return append(queries, releaseSql)
}

func jaStartTheJobnet(innerJobnetId uint64, queries ...string) []string {
	queries = append(queries, fmt.Sprintf(
		"update ja_2_run_jobnet_summary_table set load_status = %d where inner_jobnet_id = %d",
		JA_SUMMARY_JOB_STATUS_NORMAL, innerJobnetId,
	))
	return queries
}

func jaSetEndingFlagSummary(innerJobnetId uint64, endingflag int, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_jobnet_summary_table set ending_flag = %d where inner_jobnet_id = %d",
		endingflag, innerJobnetId)

	return append(queries, query)
}

func jaSetEndingFlagJobnet(innerJobnetId uint64, endingflag int, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_jobnet_table set ending_flag = %d where inner_jobnet_id = %d",
		endingflag, innerJobnetId)

	return append(queries, query)
}

func jaSetAbortFlagSummary(innerJobnetId uint64, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_jobnet_summary_table set jobnet_abort_flag = %d where inner_jobnet_id = %d AND status = %d",
		JA_JOBNET_ABORT_FLAG_ON, innerJobnetId, JA_JOBNET_STATUS_RUN)

	return append(queries, query)
}

func jaSetJobStatusSummary(innerJobnetId uint64, jobStatus JobStatusSummary, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_jobnet_summary_table set job_status = %d where inner_jobnet_id = %d",
		jobStatus, innerJobnetId)

	return append(queries, query)
}

func jaSetAbortFlagTimeout(innerJobnetId uint64, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_timeout_table set abort_flag= true where inner_jobnet_id = %d", innerJobnetId)
	return append(queries, query)
}

func jaSetEndFlagTimeout(innerJobnetId uint64, queries ...string) []string {
	query := fmt.Sprintf("update ja_2_run_timeout_table set end_flag= true where inner_jobnet_id = %d", innerJobnetId)
	return append(queries, query)
}

func jaSetRunningJobIdSummary(innerJobnetId uint64, errorJobId string, queries ...string) []string {
	query := fmt.Sprintf("UPDATE ja_2_run_jobnet_summary_table set running_job_id = '%s' where inner_jobnet_id = %d", errorJobId, innerJobnetId)
	return append(queries, query)
}

func jaSetScheduledTime(innerJobnetId uint64, scheduleTime uint64, queries ...string) []string {
	runQuery := fmt.Sprintf("UPDATE ja_2_run_jobnet_table set scheduled_time = %d where inner_jobnet_id = %d and status = %d",
		scheduleTime,
		innerJobnetId,
		JA_JOBNET_STATUS_BEGIN)
	queries = append(queries, runQuery)

	summaryQuery := fmt.Sprintf("UPDATE ja_2_run_jobnet_summary_table set scheduled_time = %d where inner_jobnet_id = %d and status = %d",
		scheduleTime,
		innerJobnetId,
		JA_JOBNET_STATUS_BEGIN)
	queries = append(queries, summaryQuery)

	return queries
}
