/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

package main

import (
	"fmt"
	"time"

	"jobarranger2/src/libs/golibs/common"
	"jobarranger2/src/libs/golibs/logger/logger"
)

func ProcessJobIconTimeout(innerJobID uint64, startTime string, processData *common.IconExecutionProcessData, nextdata *common.EventData) error {
	const funcName = "JaJobIconJobTimeout"

	logData := &logger.Logging{}
	logData, err := getLogData(processData)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200013", logger.Logging{}, funcName, err.Error())
		return err
	}

	logger.JaLog("JAICONRESULTTIMEOUT400002", *logData, funcName, innerJobID, startTime)

	iconTimeoutData, err := ExtractIconData[common.IconTimeoutData](processData.RunJobData.Data)
	if err != nil {
		message, _ := logger.JaLog("JAICONRESULTTIMEOUT200001", *logData, funcName, innerJobID, err.Error())
		return handleIconError(innerJobID, 2, message, processData, nextdata)
	}

	timeoutRunType := iconTimeoutData.TimeoutRunType
	timeout := iconTimeoutData.Timeout
	timeoutStr := time.Unix(int64(timeout), 0).Format("2006-01-02 15:04:05")
	jobID := processData.RunJobData.JobID
	jobnetID := processData.RunJobnetData.JobnetID
	execUsername := processData.RunJobnetData.ExecutionUserName

	message, _ := logger.JaLog("JAICONRESULTTIMEOUT300001", *logData, funcName, innerJobID, timeoutStr, startTime, jobnetID, jobID, execUsername)
	err = MergeAfterVariable(processData, "JOBARG_MESSAGE", message)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200006", *logData, funcName, "JOBARG_MESSAGE", message, innerJobID)
	}

	updateQuery := fmt.Sprintf(`UPDATE %s SET timeout_flag = 1`, common.Ja2RunJobTable)
	if timeoutRunType == 1 {
		updateQuery += `, method_flag = 3`
		nextdata.Event.Name = common.EventIconTimeoutStop
	} else {
		nextdata.Event.Name = common.EventIconTimeoutWarning
	}

	updateQuery += fmt.Sprintf(" WHERE inner_job_id = %d", innerJobID)
	nextdata.Queries = append(nextdata.Queries, updateQuery)

	logger.JaJobLog(common.JC_JOB_TIMEOUT, *logData)

	return nil
}

func ProcessRebootIconTimeout(innerJobID uint64, startTime string, processData *common.IconExecutionProcessData, nextdata *common.EventData) error {
	const funcName = "ProcessRebootIconTimeout"

	logData := &logger.Logging{}
	logData, err := getLogData(processData)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200013", logger.Logging{}, funcName, err.Error())
		return err
	}

	logger.JaLog("JAICONRESULTTIMEOUT400002", *logData, funcName, innerJobID, startTime)

	iconTimeoutData, err := ExtractIconData[common.IconTimeoutData](processData.RunJobData.Data)
	if err != nil {
		message, _ := logger.JaLog("JAICONRESULTTIMEOUT200001", *logData, funcName, innerJobID)
		return handleIconError(innerJobID, 2, message, processData, nextdata)
	}

	timeout := iconTimeoutData.Timeout
	timeoutStr := time.Unix(int64(timeout), 0).Format("2006-01-02 15:04:05")
	jobID := processData.RunJobData.JobID
	jobnetID := processData.RunJobnetData.JobnetID
	execUsername := processData.RunJobnetData.ExecutionUserName

	message, _ := logger.JaLog("JAICONRESULTTIMEOUT300001", *logData, funcName, innerJobID, timeoutStr, startTime, jobnetID, jobID, execUsername)
	err = MergeAfterVariable(processData, "JOBARG_MESSAGE", message)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200006", *logData, funcName, "JOBARG_MESSAGE", message, innerJobID)
	}

	updateQuery := fmt.Sprintf("UPDATE %s SET timeout_flag = 1 WHERE inner_job_id = %d", common.Ja2RunJobTable, innerJobID)

	nextdata.Event.Name = common.EventIconTimeoutWarning
	nextdata.Queries = append(nextdata.Queries, updateQuery)

	logger.JaJobLog(common.JC_JOB_TIMEOUT, *logData)

	return nil
}

func ProcessLessIconTimeout(innerJobID uint64, startTime string, processData *common.IconExecutionProcessData, nextdata *common.EventData) error {
	const funcName = "ProcessLessIconTimeout"
	logData := &logger.Logging{}
	logData, err := getLogData(processData)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200013", logger.Logging{}, funcName, err.Error())
		return err
	}

	logger.JaLog("JAICONRESULTTIMEOUT400002", *logData, funcName, innerJobID, startTime)

	iconTimeoutData, err := ExtractIconData[common.IconTimeoutData](processData.RunJobData.Data)
	if err != nil {
		message, _ := logger.JaLog("JAICONRESULTTIMEOUT200001", *logData, funcName, innerJobID)
		return handleIconError(innerJobID, 2, message, processData, nextdata)
	}

	timeout := iconTimeoutData.Timeout
	timeoutStr := time.Unix(int64(timeout), 0).Format("2006-01-02 15:04:05")
	jobID := processData.RunJobData.JobID
	jobnetID := processData.RunJobnetData.JobnetID
	execUsername := processData.RunJobnetData.ExecutionUserName

	message, _ := logger.JaLog("JAICONRESULTTIMEOUT300001", *logData, funcName, innerJobID, timeoutStr, startTime, jobnetID, jobID, execUsername)
	err = MergeAfterVariable(processData, "JOBARG_MESSAGE", message)
	if err != nil {
		logger.JaLog("JAICONRESULTNORMAL200006", *logData, funcName, "JOBARG_MESSAGE", message, innerJobID)
	}

	updateQuery := fmt.Sprintf("UPDATE %s SET timeout_flag = 1 WHERE inner_job_id = %d", common.Ja2RunJobTable, innerJobID)

	nextdata.Event.Name = common.EventIconTimeoutWarning
	nextdata.Queries = append(nextdata.Queries, updateQuery)

	logger.JaJobLog(common.JC_JOB_TIMEOUT, *logData)

	return nil
}
