/*
** Job Arranger for ZABBIX
** Copyright (C) 2025 Daiwa Institute of Research Ltd. All Rights Reserved.
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
**/

package main

import (
	"encoding/gob"
	"encoding/json"
	"fmt"
	"os"
	"unsafe"

	"jobarranger2/src/jobarg_agent/managers/agent_manager/agentutils"
	"jobarranger2/src/libs/golibs/common"
	"jobarranger2/src/libs/golibs/forker"
	"jobarranger2/src/libs/golibs/logger/logger"

	"golang.org/x/sys/windows"
)

func handleAgentFileCopy(eventData *common.EventData, netConn *common.NetConnection) error {
	var (
		protoInfo          windows.WSAProtocolInfo
		ws2_32             = windows.NewLazyDLL("ws2_32.dll")
		wsaDuplicateSocket = ws2_32.NewProc("WSADuplicateSocketW")
		agentData          common.AgentDataW
	)

	funcName := "handleAgentFileCopy"

	sc, ok := netConn.Conn.(*agentutils.SocketConn)
	if !ok {
		return fmt.Errorf("failed to type cast to agentutils.SocketConn")
	}

	// Create temporary file to share WSAProtocolInfo
	tmpFile, err := os.CreateTemp("", "wsaproto")
	if err != nil {
		return fmt.Errorf("failed to create temp file to store WSAProtocolInfo")
	}
	defer tmpFile.Close()

	proc := forker.New(forker.ProcessData{
		ExecPath: FCopyClientExecPath,
		ExecParams: []string{
			"-config-file-path", common.ConfigFilePath,
		},

		Detached:     true,
		UsePipeStdin: true,
	})

	cmd, err := proc.StartNewProcess()
	if err != nil {
		logger.WriteLog("JAAGENTMNG200022", funcName, FCopyClientExecPath, err)
		msg := fmt.Sprintf("failed to start fcopy client process [%s]: %v", FCopyClientExecPath, err)
		eventData.TCPMessage.Data = common.ResponseData{
			Result:  common.JA_JOBRESULT_FAIL,
			Message: &msg,
		}

		return agentutils.SendToServer(eventData, netConn)
	}

	logger.WriteLog("JAAGENTMNG400019", funcName, cmd.Process.Pid)

	// Prepare WSAProtocolInfo
	childPID := uint32(cmd.Process.Pid)
	ret, _, err := wsaDuplicateSocket.Call(
		uintptr(sc.S),
		uintptr(childPID),
		uintptr(unsafe.Pointer(&protoInfo)),
	)
	if ret != 0 {
		return fmt.Errorf("wsaDuplicateSocket.Call has failed")
	}

	// Send the protocol info via gob
	enc := gob.NewEncoder(tmpFile)
	err = enc.Encode(protoInfo)
	if err != nil {
		return fmt.Errorf("failed to write WSAProtocolInfo to temp file [%s]: %v", tmpFile.Name(), err)
	}

	bytes, err := json.Marshal(eventData)
	agentData.EventData = bytes
	agentData.SocketFilePath = tmpFile.Name()

	//Send event data
	agentDataBytes, err := json.Marshal(agentData)
	if err != nil {
		return fmt.Errorf("failed to marshal agent data: %v", err)
	}

	err = proc.WriteStdin(agentDataBytes)
	if err != nil {
		return fmt.Errorf("failed to write agent data into stdin pipe [raw_data=%s]: %v", string(agentDataBytes), err)
	}

	return nil
}
